﻿' RacingFighter.vb
' Single-file VB.NET Windows Forms game: "Racing Fighter"
' Paste this file into a VB.NET Windows Forms (.NET Framework or .NET) project as Form1.vb
' Or create a new project and replace the form code with this file's contents.
' Simple top-down racing shooter: player moves left/right, shoots, dodges obstacles, picks power-ups.
' Uses simple shapes (no external assets). Fully commented.

Imports System.Drawing
Imports System.Windows.Forms
Imports System.Collections.Generic

Public Module Program
    <STAThread>
    Public Sub Main()
        Application.EnableVisualStyles()
        Application.SetCompatibleTextRenderingDefault(False)
        Application.Run(New Form1())
    End Sub
End Module

Public Class Form1
    Inherits Form

    ' --- Game settings ---
    Private Const GAME_WIDTH As Integer = 480
    Private Const GAME_HEIGHT As Integer = 720
    Private Const ROAD_LEFT As Integer = 80
    Private Const ROAD_RIGHT As Integer = GAME_WIDTH - 80
    Private Const LANE_COUNT As Integer = 3

    ' Game state
    Private player As PlayerShip
    Private enemies As New List(Of Enemy)()
    Private bullets As New List(Of Bullet)()
    Private obstacles As New List(Of Obstacle)()
    Private powerUps As New List(Of PowerUp)()
    Private rand As New Random()

    Private gameTimer As Timer
    Private spawnTimer As Timer
    Private score As Integer = 0
    Private level As Integer = 1
    Private isGameOver As Boolean = False
    Private isPaused As Boolean = False

    ' Input
    Private keysDown As New HashSet(Of Keys)()

    Public Sub New()
        Me.Text = "Racing Fighter"
        Me.ClientSize = New Size(GAME_WIDTH, GAME_HEIGHT)
        Me.FormBorderStyle = FormBorderStyle.FixedSingle
        Me.MaximizeBox = False
        Me.DoubleBuffered = True
        Me.StartPosition = FormStartPosition.CenterScreen
        Me.CenterToScreen()

        ' Initialize player
        player = New PlayerShip((GAME_WIDTH \ 2) - 20, GAME_HEIGHT - 120)

        ' Main game loop timer
        gameTimer = New Timer()
        gameTimer.Interval = 16 ' ~60 FPS
        AddHandler gameTimer.Tick, AddressOf GameLoop
        gameTimer.Start()

        ' Spawn timer controls enemies/obstacles/powerups
        spawnTimer = New Timer()
        spawnTimer.Interval = 900
        AddHandler spawnTimer.Tick, AddressOf SpawnThings
        spawnTimer.Start()

        ' Input handlers
        AddHandler Me.KeyDown, AddressOf OnKeyDown
        AddHandler Me.KeyUp, AddressOf OnKeyUp
        AddHandler Me.Paint, AddressOf OnPaint
        AddHandler Me.FormClosing, AddressOf OnFormClosing

        ' Show instructions for first run in titlebar briefly
        UpdateTitle()
    End Sub

    Private Sub OnFormClosing(sender As Object, e As FormClosingEventArgs)
        gameTimer.Stop()
        spawnTimer.Stop()
    End Sub

    Private Sub UpdateTitle()
        Me.Text = $"Racing Fighter - Score: {score}  Lives: {player.Lives}  Level: {level}"
    End Sub

    Private Sub OnKeyDown(sender As Object, e As KeyEventArgs)
        keysDown.Add(e.KeyCode)

        If e.KeyCode = Keys.P Then
            isPaused = Not isPaused
        ElseIf e.KeyCode = Keys.R AndAlso isGameOver Then
            Restart()
        ElseIf e.KeyCode = Keys.Space Then
            ' Fire (space)
            If Not isGameOver AndAlso Not isPaused Then
                FireBullet()
            End If
        End If

        ' prevent ding sound
        e.SuppressKeyPress = True
    End Sub

    Private Sub OnKeyUp(sender As Object, e As KeyEventArgs)
        keysDown.Remove(e.KeyCode)
        e.SuppressKeyPress = True
    End Sub

    Private Sub Restart()
        enemies.Clear()
        bullets.Clear()
        obstacles.Clear()
        powerUps.Clear()
        score = 0
        level = 1
        player = New PlayerShip((GAME_WIDTH \ 2) - 20, GAME_HEIGHT - 120)
        isGameOver = False
        isPaused = False
        gameTimer.Start()
        spawnTimer.Start()
        UpdateTitle()
    End Sub

    Private Sub SpawnThings(sender As Object, e As EventArgs)
        If isGameOver OrElse isPaused Then Return

        ' Difficulty scales with level
        Dim enemyChance = Math.Max(25 - level * 2, 6) ' lower = more enemies
        Dim obstChance = Math.Max(35 - level * 3, 10)
        Dim powerChance = 12 ' percent

        ' Spawn enemy
        If rand.Next(100) > enemyChance Then
            Dim ex As Integer = rand.Next(ROAD_LEFT + 20, ROAD_RIGHT - 50)
            enemies.Add(New Enemy(ex, -60, Math.Min(3 + level, 9)))
        End If

        ' Spawn obstacle
        If rand.Next(100) > obstChance Then
            Dim ox As Integer = rand.Next(ROAD_LEFT + 10, ROAD_RIGHT - 40)
            obstacles.Add(New Obstacle(ox, -40, 2 + level \ 2))
        End If

        ' Spawn powerup
        If rand.Next(100) < powerChance Then
            Dim px As Integer = rand.Next(ROAD_LEFT + 20, ROAD_RIGHT - 40)
            powerUps.Add(New PowerUp(px, -40))
        End If

        ' increase level slowly
        If score > level * 200 Then
            level += 1
            spawnTimer.Interval = Math.Max(300, spawnTimer.Interval - 40)
        End If
    End Sub

    Private Sub GameLoop(sender As Object, e As EventArgs)
        If isGameOver OrElse isPaused Then
            Me.Invalidate()
            Return
        End If

        ' Player movement
        If keysDown.Contains(Keys.Left) OrElse keysDown.Contains(Keys.A) Then
            player.X -= player.Speed
        End If
        If keysDown.Contains(Keys.Right) OrElse keysDown.Contains(Keys.D) Then
            player.X += player.Speed
        End If

        ' Keep player inside road
        player.X = Math.Max(ROAD_LEFT + 8, Math.Min(player.X, ROAD_RIGHT - player.Width - 8))

        ' Update bullets
        For i = bullets.Count - 1 To 0 Step -1
            Dim b = bullets(i)
            b.Y -= b.Speed
            If b.Y < -20 Then bullets.RemoveAt(i)
        Next

        ' Update enemies
        For i = enemies.Count - 1 To 0 Step -1
            Dim en = enemies(i)
            en.Y += en.Speed
            ' slight horizontal drift
            en.X += Math.Sin((en.Y + en.Seed) / 20.0) * 2
            If en.Y > GAME_HEIGHT + 60 Then enemies.RemoveAt(i)
        Next

        ' Update obstacles
        For i = obstacles.Count - 1 To 0 Step -1
            Dim o = obstacles(i)
            o.Y += o.Speed
            If o.Y > GAME_HEIGHT + 60 Then obstacles.RemoveAt(i)
        Next

        ' Update powerups
        For i = powerUps.Count - 1 To 0 Step -1
            Dim p = powerUps(i)
            p.Y += p.Speed
            If p.Y > GAME_HEIGHT + 40 Then powerUps.RemoveAt(i)
        Next

        ' Bullet collisions with enemies & obstacles
        For bi = bullets.Count - 1 To 0 Step -1
            Dim b = bullets(bi)
            Dim hit = False
            For ei = enemies.Count - 1 To 0 Step -1
                Dim en = enemies(ei)
                If Rectangle.Intersect(b.Bounds, en.Bounds) <> Rectangle.Empty Then
                    enemies.RemoveAt(ei)
                    bullets.RemoveAt(bi)
                    score += 50
                    hit = True
                    Exit For
                End If
            Next
            If hit Then Continue For

            For oi = obstacles.Count - 1 To 0 Step -1
                Dim o = obstacles(oi)
                If Rectangle.Intersect(b.Bounds, o.Bounds) <> Rectangle.Empty Then
                    ' bullets destroy small obstacles
                    obstacles.RemoveAt(oi)
                    bullets.RemoveAt(bi)
                    score += 20
                    hit = True
                    Exit For
                End If
            Next
        Next

        ' Player collisions with enemies & obstacles
        For ei = enemies.Count - 1 To 0 Step -1
            Dim en = enemies(ei)
            If Rectangle.Intersect(player.Bounds, en.Bounds) <> Rectangle.Empty Then
                enemies.RemoveAt(ei)
                player.Lives -= 1
                If player.Lives <= 0 Then GameOver()
            End If
        Next

        For oi = obstacles.Count - 1 To 0 Step -1
            Dim o = obstacles(oi)
            If Rectangle.Intersect(player.Bounds, o.Bounds) <> Rectangle.Empty Then
                obstacles.RemoveAt(oi)
                player.Lives -= 1
                If player.Lives <= 0 Then GameOver()
            End If
        Next

        ' Powerup pickup
        For pi = powerUps.Count - 1 To 0 Step -1
            Dim p = powerUps(pi)
            If Rectangle.Intersect(player.Bounds, p.Bounds) <> Rectangle.Empty Then
                powerUps.RemoveAt(pi)
                ApplyPowerUp(p)
            End If
        Next

        ' Update score gradually
        score += 1

        UpdateTitle()
        Me.Invalidate()
    End Sub

    Private Sub GameOver()
        isGameOver = True
        ' stop spawning
        spawnTimer.Stop()
    End Sub

    Private Sub ApplyPowerUp(p As PowerUp)
        Select Case p.Type
            Case PowerUpType.ExtraLife
                player.Lives += 1
            Case PowerUpType.RapidFire
                player.FireCooldown = Math.Max(150, player.FireCooldown - 150)
                ' Revert after a while
                Dim t As New Timer()
                t.Interval = 8000
                AddHandler t.Tick, Sub(s, ea)
                                       player.FireCooldown = Math.Min(800, player.FireCooldown + 150)
                                       CType(s, Timer).Stop()
                                   End Sub
                t.Start()
            Case PowerUpType.Shield
                player.HasShield = True
                Dim t2 As New Timer()
                t2.Interval = 8000
                AddHandler t2.Tick, Sub(s, ea)
                                        player.HasShield = False
                                        CType(s, Timer).Stop()
                                    End Sub
                t2.Start()
        End Select
    End Sub

    Private lastFireTime As Integer = Environment.TickCount

    Private Sub FireBullet()
        Dim now = Environment.TickCount
        If now - lastFireTime < player.FireCooldown Then Return
        lastFireTime = now

        Dim bx = player.X + player.Width \ 2 - 4
        Dim by = player.Y - 8
        bullets.Add(New Bullet(bx, by, 10))
    End Sub

    Private Sub OnPaint(sender As Object, e As PaintEventArgs)
        Dim g = e.Graphics

        ' Draw background
        g.Clear(Color.FromArgb(30, 30, 30))

        ' Draw road
        Dim roadRect As New Rectangle(ROAD_LEFT, 0, ROAD_RIGHT - ROAD_LEFT, GAME_HEIGHT)
        Using roadBrush As New SolidBrush(Color.FromArgb(40, 40, 40))
            g.FillRectangle(roadBrush, roadRect)
        End Using

        ' Road edges
        Using pen As New Pen(Color.Gray, 4)
            g.DrawRectangle(pen, roadRect)
        End Using

        ' Lane markings
        Dim laneW = (ROAD_RIGHT - ROAD_LEFT) / LANE_COUNT
        For i = 1 To LANE_COUNT - 1
            Dim x As Single = ROAD_LEFT + i * laneW
            For y = -50 To GAME_HEIGHT Step 40
                g.FillRectangle(Brushes.White, x - 3, y + (Environment.TickCount \ 8 Mod 40), 6, 20)
            Next
        Next

        ' Draw player
        player.Draw(g)

        ' draw bullets
        For Each b In bullets
            b.Draw(g)
        Next

        ' draw enemies
        For Each en In enemies
            en.Draw(g)
        Next

        ' draw obstacles
        For Each o In obstacles
            o.Draw(g)
        Next

        ' draw powerups
        For Each p In powerUps
            p.Draw(g)
        Next

        ' HUD
        Using sf As New StringFormat()
            sf.Alignment = StringAlignment.Near
            sf.LineAlignment = StringAlignment.Near
            g.DrawString($"Score: {score}", New Font("Consolas", 12, FontStyle.Bold), Brushes.White, 10, 8)
            g.DrawString($"Lives: {player.Lives}", New Font("Consolas", 12, FontStyle.Bold), Brushes.White, 120, 8)
            g.DrawString($"Level: {level}", New Font("Consolas", 12, FontStyle.Bold), Brushes.White, 220, 8)
        End Using

        ' Pause / Game Over overlay
        If isPaused Then
            DrawOverlay(g, "PAUSED - Press P to resume")
        ElseIf isGameOver Then
            DrawOverlay(g, $"GAME OVER  Score: {score}" + vbCrLf + "Press R to restart")
        End If
    End Sub

    Private Sub DrawOverlay(g As Graphics, text As String)
        Using b As New SolidBrush(Color.FromArgb(180, Color.Black))
            g.FillRectangle(b, 0, 0, GAME_WIDTH, GAME_HEIGHT)
        End Using
        Using f As New Font("Arial", 22, FontStyle.Bold)
            Dim sz = g.MeasureString(text, f)
            g.DrawString(text, f, Brushes.White, (GAME_WIDTH - sz.Width) / 2, (GAME_HEIGHT - sz.Height) / 2)
        End Using
    End Sub
End Class

' --- Game entity classes ---
Public Class PlayerShip
    Public Property X As Integer
    Public Property Y As Integer
    Public ReadOnly Property Width As Integer = 40
    Public ReadOnly Property Height As Integer = 60
    Public Property Speed As Integer = 6
    Public Property Lives As Integer = 3
    Public Property FireCooldown As Integer = 400 ' ms
    Public Property HasShield As Boolean = False

    Public Sub New(x As Integer, y As Integer)
        Me.X = x
        Me.Y = y
    End Sub

    Public ReadOnly Property Bounds As Rectangle
        Get
            Return New Rectangle(X, Y, Width, Height)
        End Get
    End Property

    Public Sub Draw(g As Graphics)
        ' ship body
        Dim bodyPoly As Point() = {New Point(X + Width \ 2, Y), New Point(X + Width, Y + Height \ 2), New Point(X + Width \ 2, Y + Height), New Point(X, Y + Height \ 2)}
        g.FillPolygon(Brushes.DodgerBlue, bodyPoly)
        g.DrawPolygon(Pens.Black, bodyPoly)

        ' cockpit
        g.FillEllipse(Brushes.LightBlue, X + 10, Y + 10, 20, 20)

        ' shield
        If HasShield Then
            Using pen As New Pen(Color.FromArgb(120, Color.Cyan), 4)
                g.DrawEllipse(pen, X - 8, Y - 8, Width + 16, Height + 16)
            End Using
        End If
    End Sub
End Class

Public Class Enemy
    Public Property X As Integer
    Public Property Y As Integer
    Public Property Speed As Integer
    Public Property Width As Integer = 44
    Public Property Height As Integer = 50
    Public Property Seed As Integer

    Public Sub New(x As Integer, y As Integer, speed As Integer)
        Me.X = x
        Me.Y = y
        Me.Speed = speed
        Me.Seed = (New Random()).Next(1000)
    End Sub

    Public ReadOnly Property Bounds As Rectangle
        Get
            Return New Rectangle(CInt(X), CInt(Y), Width, Height)
        End Get
    End Property

    Public Sub Draw(g As Graphics)
        g.FillRectangle(Brushes.IndianRed, X, Y, Width, Height)
        g.DrawRectangle(Pens.Black, X, Y, Width, Height)
        g.DrawString("EN", New Font("Consolas", 10, FontStyle.Bold), Brushes.Black, X + 6, Y + 14)
    End Sub
End Class

Public Class Bullet
    Public Property X As Integer
    Public Property Y As Integer
    Public Property Speed As Integer = 10
    Public ReadOnly Property Width As Integer = 8
    Public ReadOnly Property Height As Integer = 18

    Public Sub New(x As Integer, y As Integer, speed As Integer)
        Me.X = x
        Me.Y = y
        Me.Speed = speed
    End Sub

    Public ReadOnly Property Bounds As Rectangle
        Get
            Return New Rectangle(X, Y, Width, Height)
        End Get
    End Property

    Public Sub Draw(g As Graphics)
        g.FillRectangle(Brushes.Yellow, X, Y, Width, Height)
        g.DrawRectangle(Pens.Orange, X, Y, Width, Height)
    End Sub
End Class

Public Class Obstacle
    Public Property X As Integer
    Public Property Y As Integer
    Public Property Speed As Integer = 3
    Public Property Width As Integer = 36
    Public Property Height As Integer = 36

    Public Sub New(x As Integer, y As Integer, speed As Integer)
        Me.X = x
        Me.Y = y
        Me.Speed = speed
    End Sub

    Public ReadOnly Property Bounds As Rectangle
        Get
            Return New Rectangle(X, Y, Width, Height)
        End Get
    End Property

    Public Sub Draw(g As Graphics)
        g.FillEllipse(Brushes.SaddleBrown, X, Y, Width, Height)
        g.DrawEllipse(Pens.Black, X, Y, Width, Height)
    End Sub
End Class

Public Enum PowerUpType
    ExtraLife
    RapidFire
    Shield
End Enum

Public Class PowerUp
    Public Property X As Integer
    Public Property Y As Integer
    Public Property Speed As Integer = 2
    Public Property Width As Integer = 26
    Public Property Height As Integer = 26
    Public Property Type As PowerUpType

    Private Shared rnd As New Random()

    Public Sub New(x As Integer, y As Integer)
        Me.X = x
        Me.Y = y
        ' random type
        Dim r = rnd.Next(100)
        If r < 40 Then
            Type = PowerUpType.ExtraLife
        ElseIf r < 75 Then
            Type = PowerUpType.RapidFire
        Else
            Type = PowerUpType.Shield
        End If
    End Sub

    Public ReadOnly Property Bounds As Rectangle
        Get
            Return New Rectangle(X, Y, Width, Height)
        End Get
    End Property

    Public Sub Draw(g As Graphics)
        Select Case Type
            Case PowerUpType.ExtraLife
                g.FillEllipse(Brushes.Pink, X, Y, Width, Height)
                g.DrawString("+1", New Font("Consolas", 8, FontStyle.Bold), Brushes.White, X + 4, Y + 6)
            Case PowerUpType.RapidFire
                g.FillEllipse(Brushes.LimeGreen, X, Y, Width, Height)
                g.DrawString("RF", New Font("Consolas", 8, FontStyle.Bold), Brushes.Black, X + 4, Y + 6)
            Case PowerUpType.Shield
                g.FillEllipse(Brushes.Cyan, X, Y, Width, Height)
                g.DrawString("SH", New Font("Consolas", 8, FontStyle.Bold), Brushes.Black, X + 4, Y + 6)
        End Select
        g.DrawEllipse(Pens.Black, X, Y, Width, Height)
    End Sub
End Class
